<?php

namespace WPaaS\Admin;

use WPaaS\API_Interface;
use \WPaaS\Helpers;
use \WPaaS\Plugin;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

final class Nps_Form
{
    use \WPaaS\Helpers;

    const CACHING_INTERVAL = 21600; // 6 hours

    const OPTION_NAME = 'wpaas_nps_form_%s';

    /**
     * Instance of the API.
     *
     * @var API_Interface
     */
    private API_Interface $api;

    private string $npsFormId = 'wptool#nps-system-plugin-dashboard';

    /**
     * @var bool
     */
    private bool $has_api_failed = false;

    public function __construct(API_Interface $api)
    {
        $this->api = $api;
        add_action( 'init', [ $this, 'init' ] );
    }

    public function init() {
            $shoud_show = is_admin() &&
                current_user_can( 'administrator' ) &&
                ! Plugin::is_staging_site() &&
                defined( 'GD_RUM_ENABLED' ) &&
                GD_RUM_ENABLED &&
                Plugin::is_gd() &&
                ( ! defined( 'DOING_AJAX' ) || ! DOING_AJAX );

            if(!$shoud_show ) {
                return;
            }

            if (!$GLOBALS['wpaas_feature_flag']->get_feature_flag_value( 'nps_90_days', false )) {
                return;
            }

            if (!$this->should_show_nps_form()) {
                return;
            }
            add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    /**
     * Check if NPS form should be shown with caching.
     *
     * @return bool
     */
    private function should_show_nps_form(): bool {
        if ( !defined('GD_ACCOUNT_UID') ) {
            return false;
        }

        $option_name = sprintf( self::OPTION_NAME, GD_ACCOUNT_UID );
        $cached_data = get_option( $option_name );

        // No cache exists - fetch from API
        if ( false === $cached_data ) {
            return $this->fetch_and_cache_nps_form_status( $option_name, true );
        }

        // Cache expired - refresh from API
        if ( empty( $cached_data['timestamp'] ) || time() - $cached_data['timestamp'] > self::CACHING_INTERVAL ) {
            $result = $this->fetch_and_cache_nps_form_status( $option_name, false );

            // If API failed, reuse cached value and update timestamp to avoid hammering
            if ( $this->has_api_failed && isset( $cached_data['show_form'] ) ) {
                $cached_data['timestamp'] = time();
                update_option( $option_name, $cached_data );
                return $cached_data['show_form'];
            }

            return $result;
        }

        // Return cached value
        return isset( $cached_data['show_form'] ) ? $cached_data['show_form'] : false;
    }

    /**
     * Fetch NPS form status from API and cache the result.
     *
     * @param string $option_name
     * @param bool   $first_fetch
     *
     * @return bool
     */
    private function fetch_and_cache_nps_form_status( string $option_name, bool $first_fetch ): bool {
        $result = $this->api->show_nps_form( $this->npsFormId );

        if ( null === $result ) {
            $this->has_api_failed = true;

            // On first fetch with API failure, cache false as default
            if ( $first_fetch ) {
                $cached_data = [
                    'show_form' => false,
                    'timestamp' => time(),
                ];
                update_option( $option_name, $cached_data );
            }

            return false;
        }

        $cached_data = [
            'show_form' => (bool) $result,
            'timestamp' => time(),
        ];
        update_option( $option_name, $cached_data );

        return (bool) $result;
    }

    public function enqueue_assets()
    {
        if ( defined( 'GD_SITE_CREATED' ) ) {
            $siteCreationDate = ( new  \DateTime() )->setTimestamp( GD_SITE_CREATED );
        }

        wp_enqueue_style('wpaas-nps-style', self::get_cdn_url() . 'standalone/feedback-widget-vanilla.css');
        wp_enqueue_script('wpaas-nps', self::get_cdn_url() . 'standalone/feedback-widget-vanilla.js');
        wp_localize_script('wpaas-nps', 'wpaas_nps_object', [
            'getConfigUrl' => add_query_arg(
                'nps_form_id',
                rawurlencode($this->npsFormId), // let add_query_arg handle encoding
                rest_url( 'wpaas/v1/nps-form-configuration' )
            ),
            'dismissUrl' => add_query_arg( 'nps_form_id', rawurlencode($this->npsFormId), rest_url( 'wpaas/v1/nps-form-dismiss' ) ),
            'submitUrl' => add_query_arg( 'nps_form_id', rawurlencode($this->npsFormId), rest_url( 'wpaas/v1/nps-form-submit' ) ),
            'localeBaseUrl' => self::get_cdn_url() . "assets/locales",
            'locale' => get_locale(),
            'nonce' => wp_create_nonce( 'wp_rest' ),
            'metaData' =>  json_encode( [
                    'customerId'             => defined( 'GD_CUSTOMER_ID' ) ? GD_CUSTOMER_ID : null,
                    'guid'                   => defined( 'GD_ACCOUNT_UID' ) ? GD_ACCOUNT_UID : null,
                    'productId'              => defined( 'GD_ACCOUNT_UID' ) ? GD_ACCOUNT_UID : null,
                    'product_name'           => 'MWP',
                    'mwpSystemPluginVersion' => Plugin::version(),
                    'wpUserId'               => get_current_user_id(),
                    'wpVersion'              => get_bloginfo( 'version' ),
                    'mwpPlanName'            => defined( 'GD_PLAN_NAME' ) ? GD_PLAN_NAME : null,
                    'wpLocale'               => get_locale(),
                    'woocommerceVersion'     => defined( 'WC_VERSION' ) ? WC_VERSION : null,
                    'siteCreatedAt'          => defined( 'GD_SITE_CREATED' ) ? $siteCreationDate->format( \DateTime::ATOM ) : null,
                    'siteAgeDays'            => defined( 'GD_SITE_CREATED' ) ? floor( ( time() - GD_SITE_CREATED ) / 86400 ) : 0,
                    'isV2App'                => self::is_wpaas_v2()
                ]),
            'feedbackType' => 'NPS'
        ]);
    }

    public function get_cdn_url(): string
    {
        $env = Plugin::get_env();

        if ( in_array( $env, [ 'dev', 'test' ], true ) ) {
            return "https://d2mt2uivr8ua2g.cloudfront.net/";
        }

        return 'https://ds9ywulh7jrls.cloudfront.net/';
    }
}